/*:
 * @target MZ
 * @plugindesc v1.1【HSアドオン】セーブ後のSpine再生を確実化＋デバッグログ/オーバーレイ
 * @author HS
 * @help
 * ■ このプラグインが行うこと（再掲）
 * ・オプション重ね表示などでシーン遷移せずにセーブした後でも、
 *   マップへ戻ったタイミングで PKD_Spine2D の立ち絵アニメを必ず「再生状態」に戻します。
 * ・HS_SpineSnapshotRestore.js の再開ロジック（Scene_Map.update 内）に
 *   「再開予約フラグ」を渡して起動させます。
 *
 * ■ デバッグ機能（本バージョンで追加）
 * ・コンソールへ詳細ログを出力（ON/OFF可能）
 * ・画面左上に小さなデバッグオーバーレイ表示（ON/OFF可能）
 * ・現在のSpineノード推定一覧をコンソールにダンプするコマンドを追加
 *
 * ■ 導入順（推奨）
 *  1) PKD_Spine2DPlayer.js
 *  2) HS_SpineSnapshotRestore.js
 *  3) HS_SpineAfterLoadDedupe.js（使っていれば）
 *  4) HS_SpineResumeOnSave.js   ← このプラグイン
 *  5) HS_OptionsOverlay.js / HS_OptionsOverlay_InputLock.js
 *
 * ■ パラメータ
 * @param resumeWaitFrames
 * @text セーブ後の再開待機フレーム
 * @type number
 * @min 0
 * @max 60
 * @default 0
 * @desc セーブから戻った後、再開処理を走らせるまでの待機フレーム数。0で次フレームに即再開。
 *
 * @param debugConsole
 * @text コンソールにデバッグ出力
 * @type boolean
 * @default true
 *
 * @param debugOverlay
 * @text 画面左上にデバッグオーバーレイ
 * @type boolean
 * @default false
 *
 * ■ プラグインコマンド
 * @command ToggleDebug
 * @text デバッグ出力のON/OFFを切替
 * @arg console
 * @text コンソール出力をON
 * @type boolean
 * @default true
 * @arg overlay
 * @text デバッグオーバーレイをON
 * @type boolean
 * @default false
 *
 * @command DumpSpine
 * @text Spine状態をコンソール出力
 *
 * @command ResumeSoon
 * @text 指定フレーム後に再開を予約
 * @arg frames
 * @type number
 * @default 0
 * @min 0
 *
 * ■ ライセンス
 * ・本プラグインはプロジェクト内で自由に改変・配布して構いません。
 */
(() => {
  "use strict";
  const PN = "HS_SpineResumeOnSave";
  const P  = PluginManager.parameters(PN);
  const WAIT = Math.max(0, Number(P.resumeWaitFrames || 0));

  let DEBUG_CONSOLE = String(P.debugConsole || "true") === "true";
  let DEBUG_OVERLAY = String(P.debugOverlay || "false") === "true";

  const sys = ()=> $gameSystem;
  const hasPlugin = name => !!PluginManager._scripts?.some(s => String(s).toLowerCase() === String(name).toLowerCase());

  // ─────────────────────────────────────────────
  // ログユーティリティ
  // ─────────────────────────────────────────────
  const tag = "[HS_SpineResumeOnSave]";
  const log  = (...a)=>{ if (DEBUG_CONSOLE && window.console) console.log(tag, ...a); };
  const warn = (...a)=>{ if (DEBUG_CONSOLE && window.console) console.warn(tag, ...a); };
  const err  = (...a)=>{ if (DEBUG_CONSOLE && window.console) console.error(tag, ...a); };

  function initResumeFlags(){
    if (!sys()) return;
    sys()._hsPendResumeAfterLoad ??= false;
    sys()._hsResumeWait        ??= 0;
  }

  // ─────────────────────────────────────────────
  // 可能ならDevToolsを開ける（NW.jsのみ）
  // ─────────────────────────────────────────────
  try{
    if (DEBUG_CONSOLE && Utils.isNwjs() && !window._hsOpenedDevTools){
      window._hsOpenedDevTools = true;
      nw.Window.get().showDevTools();
      log("NW.js DevTools を開きました");
    }
  }catch(_){}

  // ─────────────────────────────────────────────
  // セーブ成功時：再開予約をセット
  // ─────────────────────────────────────────────
  if (typeof Scene_Save !== "undefined") {
    const _onSaveOk = Scene_Save.prototype.onSaveSuccess;
    Scene_Save.prototype.onSaveSuccess = function(){
      initResumeFlags();
      const before = { pend: sys()._hsPendResumeAfterLoad, wait: sys()._hsResumeWait };
      sys()._hsPendResumeAfterLoad = true;
      sys()._hsResumeWait = WAIT;
      log("セーブ成功 → 再開を予約", { before, after:{ pend: true, wait: WAIT } });
      _onSaveOk.call(this);
    };
  }

  // ─────────────────────────────────────────────
  // セーブ/ロード画面から戻る瞬間：Scene_Map に待機フレームを植える
  // ─────────────────────────────────────────────
  if (typeof Scene_File !== "undefined") {
    const _pop = Scene_File.prototype.popScene;
    Scene_File.prototype.popScene = function(){
      _pop.call(this); // 前のシーンへ戻る
      try {
        const sc = SceneManager._scene;
        log("Scene_File.popScene 復帰先:", sc?.constructor?.name || sc);
        if (sys()?.["_hsPendResumeAfterLoad"]) {
          if (sc instanceof Scene_Map) {
            sc._hsResumeWait = (sys()._hsResumeWait ?? 0);
            log("Scene_Map に再開待機を植え付け:", sc._hsResumeWait);
          } else {
            warn("復帰先が Scene_Map ではありません。再開は次のMap復帰で消費されます。");
          }
        } else {
          log("再開予約フラグは立っていません（他プラグインで既に消費？）");
        }
      } catch(e) { err("popScene後の処理で例外:", e); }
    };
  }

  // ─────────────────────────────────────────────
  // デバッグ：Spineノード探索（推定）
  // ─────────────────────────────────────────────
  function findSpineNodes(root, out = []){
    try{
      if (!root) return out;
      const hasSpineTrait = (o)=>{
        if (!o) return false;
        // よくある特徴：pixi-spine/PKDラッパで見られるプロパティ
        return !!(o.spine || o.skeleton || o.state || o._spine || o._skeleton || o.setAnimation);
      };
      const children = root.children || root._children;
      if (hasSpineTrait(root)) out.push(root);
      if (children?.length){
        for (let i=0;i<children.length;i++){
          findSpineNodes(children[i], out);
        }
      }
    }catch(e){}
    return out;
  }

  function dumpSpineFromScene(scene){
    try{
      const tgt = scene?._spriteset || scene;
      const list = findSpineNodes(tgt);
      log("Spineノード推定数:", list.length);
      list.forEach((n,i)=>{
        const info = {
          idx:i,
          name: (n.name||n._name||n.constructor?.name),
          timeScale: (n.timeScale ?? n.state?.timeScale ?? n.spine?.state?.timeScale),
          hasState: !!(n.state || n.spine?.state),
          tracks: safeTracks(n),
        };
        log(" -", info);
      });
      if (list.length===0) warn("Spineと思しきノードは見つかりませんでした。描画親や表示レイヤーを再確認してください。");
    }catch(e){ err("dumpSpineFromScene 例外:", e); }
  }

  function safeTracks(n){
    try{
      const st = n.state || n.spine?.state;
      if (!st) return null;
      const tracks = st.tracks || st._tracks || [];
      return tracks.map((t,ti)=>{
        try{
          if (!t) return null;
          return {
            ti,
            anim: t.animation?.name,
            loop: !!t.loop,
            trackTime: t.trackTime,
            delay: t.delay,
            timeScale: t.timeScale,
          };
        }catch(_){ return null; }
      });
    }catch(_){ return null; }
  }

  // ─────────────────────────────────────────────
  // 画面左上オーバーレイ
  // ─────────────────────────────────────────────
  class HS_SpineResumeOverlay extends Sprite {
    constructor(){
      super(new Bitmap(360, 96));
      this.x = 6; this.y = 6;
      this.z = 9e6;
      this._tick = 0;
      this.bitmap.fontSize = 18;
      this.bitmap.outlineColor = "black";
      this.bitmap.outlineWidth = 3;
      this.opacity = 215;
      this.refresh(true);
    }
    update(){
      super.update();
      if ((this._tick++ % 15) === 0) this.refresh(false);
    }
    refresh(force){
      const bm = this.bitmap;
      bm.clear();
      const s = sys();
      const sc = SceneManager._scene;
      const pend = s?._hsPendResumeAfterLoad;
      const gWait = s?._hsResumeWait;
      const mWait = sc?._hsResumeWait;
      let spineCount = 0;
      try { spineCount = findSpineNodes(sc?._spriteset || sc).length; } catch(_){}
      const lines = [
        "HS_SpineResumeOnSave [Debug]",
        `Pend:${pend?'T':'F'}  GWait:${gWait}  MWait:${mWait}`,
        `Scene:${sc?.constructor?.name||'?' }  SpineNodes:${spineCount}`,
        hasPlugin("HS_SpineSnapshotRestore") ? "Restore: FOUND" : "Restore: NOT FOUND",
      ];
      for (let i=0;i<lines.length;i++){
        bm.drawText(lines[i], 0, i*22, bm.width, 24, "left");
      }
    }
  }

  // Scene_Map にオーバーレイを載せる
  if (typeof Scene_Map !== "undefined") {
    const _createDisp = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function(){
      _createDisp.call(this);
      if (DEBUG_OVERLAY){
        this._hsSpineResumeOverlay = new HS_SpineResumeOverlay();
        this.addChild(this._hsSpineResumeOverlay);
        log("デバッグオーバーレイ生成");
      }
    };
  }

  // ─────────────────────────────────────────────
  // プラグインコマンド
  // ─────────────────────────────────────────────
  PluginManager.registerCommand(PN, "ToggleDebug", args => {
    DEBUG_CONSOLE = String(args.console || "true") === "true";
    const ov = String(args.overlay || "false") === "true";
    DEBUG_OVERLAY = ov;
    const sc = SceneManager._scene;
    // 反映
    if (sc instanceof Scene_Map){
      if (ov && !sc._hsSpineResumeOverlay){
        sc._hsSpineResumeOverlay = new HS_SpineResumeOverlay();
        sc.addChild(sc._hsSpineResumeOverlay);
      }
      if (!ov && sc._hsSpineResumeOverlay){
        sc.removeChild(sc._hsSpineResumeOverlay);
        sc._hsSpineResumeOverlay.destroy({children:true});
        sc._hsSpineResumeOverlay = null;
      }
    }
    log("ToggleDebug:", { DEBUG_CONSOLE, DEBUG_OVERLAY });
  });

  PluginManager.registerCommand(PN, "DumpSpine", _args => {
    const sc = SceneManager._scene;
    log("DumpSpine: 現在のシーン:", sc?.constructor?.name);
    dumpSpineFromScene(sc);
  });

  PluginManager.registerCommand(PN, "ResumeSoon", args => {
    const f = Math.max(0, Number(args.frames||0));
    window.HS?.resumeSpineSoon?.(f);
  });

  // ─────────────────────────────────────────────
  // 任意：他プラグインから明示的に「すぐ再開」を予約するヘルパー
  // ─────────────────────────────────────────────
  window.HS = window.HS || {};
  window.HS.resumeSpineSoon = function(frames = WAIT){
    try {
      initResumeFlags();
      const before = { pend: sys()._hsPendResumeAfterLoad, wait: sys()._hsResumeWait };
      sys()._hsPendResumeAfterLoad = true;
      sys()._hsResumeWait = Math.max(0, frames|0);
      const sc = SceneManager._scene;
      if (sc instanceof Scene_Map) sc._hsResumeWait = sys()._hsResumeWait;
      log("resumeSpineSoon 呼び出し", { before, after:{ pend:true, wait: sys()._hsResumeWait } });
    } catch(e) { err("resumeSpineSoon 例外:", e); }
  };

  // ─────────────────────────────────────────────
  // 起動時の環境チェック
  // ─────────────────────────────────────────────
  log("起動", {
    waitDefault: WAIT,
    hasRestore: hasPlugin("HS_SpineSnapshotRestore"),
    hasPKD: hasPlugin("PKD_Spine2DPlayer"),
  });
})();

